/*
Author: Radek Henys (admin@mouseviator.com)
Version: 1.0b
Last update: 26.3.2014
License: LGPL v3
*/

#include "common.h"
#include <TlHelp32.h>
#include <Psapi.h>
#include "process.h"

#pragma comment(lib, "psapi.lib") // Added to support GetProcessMemoryInfo()

//added to support PROCESS_ALL_ACCESS for OpenProcess on Windows XP
#define _WIN32_WINNT _WIN32_WINNT_WINXP

/*
	This function will return process handle for first process matching given name or process id.

	Parameters:
		process_name - name of the process we want to get process id for. Can be part of name.
		processid - id of process we want to get handle for.
		dwDesiredAccess - access rights we want to have to process. See http://msdn.microsoft.com/en-us/library/windows/desktop/ms684880(v=vs.85).aspx for more details.
	Returns:
		Handle of process matching given name or with given id.
	Throws:
		Will throw error if something goes wrong.
*/
int getHandle(lua_State *L) {
	//get first parameter, that should be the name of the process
	std::string process_name = "";
	std::string exeName = "";
	long argument;
	DWORD processID = INVALID_PROCESSID_VALUE;
	DWORD dwDesiredAccess = PROCESS_QUERY_INFORMATION | PROCESS_SET_INFORMATION | PROCESS_VM_READ;
	HANDLE processHandle = NULL;

	//how many arguments we have
	int arg_count = lua_gettop(L);

	//check which type of argument did we receive, is it process id or name?
	if (lua_isnumber(L, 1)) {
		argument = luaL_checklong(L, 1);
		if (argument < 0) {
			return luaL_error(L, "Invalid argument. A process id given: %d. Process ID cannot be negative!", argument);
		}
		else {
			//store read argument as processID
			processID = (DWORD)argument;
		}
	}
	else if (lua_isstring(L, 1)) {
		process_name = luaL_checkstring(L, 1);
	}

	//check if second argument was given, this should be access rights
	if (arg_count == 2 && lua_isnumber(L, 2)) {
		argument = luaL_checklong(L, 2);
		if (argument < 0) {
			return luaL_error(L, "Invalid argument. Access rights given: %d. Access rights cannot be negative!", argument);
		}
		else {
			//store read argument as processID
			dwDesiredAccess = (DWORD)argument;
		}
	}

	PROCESSENTRY32 entry;
	entry.dwSize = sizeof(PROCESSENTRY32);

	HANDLE snapshot = CreateToolhelp32Snapshot(TH32CS_SNAPPROCESS, NULL);
	if (snapshot == INVALID_HANDLE_VALUE) {
		return getLastError(L);
	}

	//iterate through processes to find one with matching name
	if (Process32First(snapshot, &entry) == TRUE)
	{
		while (Process32Next(snapshot, &entry) == TRUE)
		{
			if (processID != INVALID_PROCESSID_VALUE) {
				if (entry.th32ProcessID == processID) {
					//get process handle
					processHandle = OpenProcess(dwDesiredAccess, FALSE, entry.th32ProcessID);
					break;
				}
			}
			else {
				exeName = entry.szExeFile;
				if (exeName.find(process_name) != std::string::npos)
				{
					//get process handle
					processHandle = OpenProcess(dwDesiredAccess, FALSE, entry.th32ProcessID);
					break;
				}
			}
		}
	}

	CloseHandle(snapshot);

	//return process handle
	if (processHandle == NULL) {
		//there was an error opening process, get it
		return getLastError(L);
	}
	else {
		lua_pushinteger(L, (long)processHandle);
	}

	//how many values we left on stack
	return 1;
}


/*
	This function will close handle opened by getHandle.

	Parameters:
		handle - a handle to process opened by getHandle.
	Returns:
		Nothing if there were no errors. Otherwise, raises error.
	Throws:
		Will throw error if something goes wrong.
*/
int closeHandle(lua_State *L) {
	long handle;

	//check if given parameter is number
	handle = luaL_checklong(L, 1);
	if (handle < 0) {
		return luaL_error(L, "Invalid argument. A process handle given: %d. Process handle cannot be negative!", handle);
	}

	BOOL result = CloseHandle((HWND)(LONG_PTR)handle);
	if (result == FALSE) {
		return getLastError(L);
	}

	return 0;
}

/*
	This function will return process id for first process matching the given name or the process 	with given handle. If no parameters are given,  the id for current process is returned. 

	Parameters:
		process_name - name of the process we want to get process id for. Can be part of name.
		handle - handle of process we want to get process id for.
	Returns:
		ID of process matching given name or with given handle.
	Throws:
		Will throw error if something goes wrong.
*/
int getProcessID(lua_State *L) {
	DWORD processId = 0;
	//if no argument given, get process id for current process
	if (lua_gettop(L) == 0) {
		processId = GetCurrentProcessId();
		//there is no mention in the documentation that above function could fail in any way,
		//so just return the result
		lua_pushinteger(L, processId);
	}
	else {
		//We accept argument types of string - process name, or number - handle of the process
		if (lua_isnumber(L, 1)) {
			//find id by process handle
			long argument;
			HANDLE hProcess;

			//check if given parameter is number
			argument = luaL_checklong(L, 1);
			if (argument < 0) {
				return luaL_error(L, "Invalid argument. A process handle given: %d. Process handle cannot be negative!", argument);
			}
			else {
				//store read argument as processID
				hProcess = &argument;
			}

			processId = GetProcessId(hProcess);
			if (processId == 0) {
				//id cannot be 0, we failed
				lua_pushnil(L);
			}
			else {
				lua_pushinteger(L, processId);
			}
		}
		else if (lua_isstring(L, 1)) {
			//get first parameter, that should be the name of the process
			std::string process_name = luaL_checkstring(L, 1);
			std::string exeName;
			long processID = INVALID_PROCESSID_VALUE;

			PROCESSENTRY32 entry;
			entry.dwSize = sizeof(PROCESSENTRY32);

			HANDLE snapshot = CreateToolhelp32Snapshot(TH32CS_SNAPPROCESS, NULL);
			if (snapshot == INVALID_HANDLE_VALUE) {
				return getLastError(L);
			}

			//iterate through processes to find one with matching name
			if (Process32First(snapshot, &entry) == TRUE)
			{
				while (Process32Next(snapshot, &entry) == TRUE)
				{
					exeName = entry.szExeFile;
					if (exeName.find(process_name) != std::string::npos)
					{
						//save process ID to return it
						processID = (DWORD)entry.th32ProcessID;
						break;
					}
				}
			}

			CloseHandle(snapshot);

			//return process id
			if (processID == INVALID_PROCESSID_VALUE) {
				lua_pushnil(L);
			}
			else {
				lua_pushinteger(L, processID);
			}
		}
		else {
			return luaL_error(L, "Unsupported argument type: %d. Only process name (string) or handle (number) are supported!", lua_type(L, 1));
		}
	}

	//how many values we left on stack
	return 1;
}

/*
	This function will return an array of process ids matching the given name.

	Parameters:
		process_name - name of the process we want to get process id for. Can be part of name.
	Returns:
		IDS of all processes matching given name.
	Throws:
		Will throw error if something goes wrong.
*/
int getProcessIDs(lua_State *L) {
	//get first parameter, that should be the name of the process
	std::string process_name = luaL_checkstring(L, 1);
	std::string exeName;
	int index = 1;

	PROCESSENTRY32 entry;
	entry.dwSize = sizeof(PROCESSENTRY32);

	HANDLE snapshot = CreateToolhelp32Snapshot(TH32CS_SNAPPROCESS, NULL);
	if (snapshot == INVALID_HANDLE_VALUE) {
		return getLastError(L);
	}

	//create table on stack for results
	lua_newtable(L);

	//iterate through processes to find one with matching name
	if (Process32First(snapshot, &entry) == TRUE)
	{
		while (Process32Next(snapshot, &entry) == TRUE)
		{
			exeName = entry.szExeFile;
			if (exeName.find(process_name) != std::string::npos)
			{
				//save process ID to return it
				lua_pushinteger(L, entry.th32ProcessID);
				//results are in table, set t[index] = processID
				//this actually pushes index value to second position from top of the stack
				lua_rawseti(L, -2, index++);
			}
		}
	}

	CloseHandle(snapshot);

	//how many values we left on stack, one table
	return 1;
}

/*
	This function retrieves the amount of memory private bytes the process specified by handle is using.

	Parameters:
		handle - handle of process we want to get affinity for. This has to be greater than 0. We treat that as long.
	Returns:
		privatebytes - the amount of memory private bytes the process is using.
	Throws:
		Will throw error if something goes wrong.
*/
int getPrivateBytes(lua_State *L) {
	long argument;
	long handle;
	PROCESS_MEMORY_COUNTERS_EX pmc;

	//check if given parameter is number
	argument = luaL_checklong(L, 1);
	if (argument < 0) {
		return luaL_error(L, "Invalid argument. A process handle given: %d. Process handle cannot be negative!", argument);
	}
	else {
		//store read argument as process handle
		handle = argument;
	}

	//get process memory info
	BOOL result = GetProcessMemoryInfo((HWND)(LONG_PTR)handle, (PROCESS_MEMORY_COUNTERS*)&pmc, sizeof(pmc));
	//if call failed, try to get last code
	if (result == FALSE) {
		//get last error code and terminate execution of this function 
		return getLastError(L);
	}

	//push result onto stack
	lua_pushinteger(L, pmc.PrivateUsage);

	//how many values we left on stack
	return 1;
}

/*
	This function retrieves memory usage data of the process specified by handle.

	Parameters:
		handle - handle of process we want to get affinity for. This has to be greater than 0. We treat that as long.
	Returns:
		A table with memory usage informations of specified process.
	Throws:
		Will throw error if something goes wrong.
*/
int getMemoryInfo(lua_State *L) {
	long argument;
	long handle;
	PROCESS_MEMORY_COUNTERS_EX pmc;

	//check if given parameter is number
	argument = luaL_checklong(L, 1);
	if (argument < 0) {
		return luaL_error(L, "Invalid argument. A process handle given: %d. Process handle cannot be negative!", argument);
	}
	else {
		//store read argument as process handle
		handle = argument;
	}

	//get process memory info
	BOOL result = GetProcessMemoryInfo((HWND)(LONG_PTR)handle, (PROCESS_MEMORY_COUNTERS*)&pmc, sizeof(pmc));
	//if call failed, try to get last code
	if (result == FALSE) {
		//get last error code and terminate execution of this function 
		return getLastError(L);
	}

	lua_newtable(L);
	//push result onto stack
	lua_pushstring(L, "PageFaultCount");
	lua_pushinteger(L, pmc.PageFaultCount);
	lua_settable(L, -3);

	lua_pushstring(L, "PagefileUsage");
	lua_pushinteger(L, pmc.PagefileUsage);
	lua_settable(L, -3);

	lua_pushstring(L, "PeakPagefileUsage");
	lua_pushinteger(L, pmc.PeakPagefileUsage);
	lua_settable(L, -3);

	lua_pushstring(L, "PeakWorkingSetSize");
	lua_pushinteger(L, pmc.PeakWorkingSetSize);
	lua_settable(L, -3);

	lua_pushstring(L, "PrivateUsage");
	lua_pushinteger(L, pmc.PrivateUsage);
	lua_settable(L, -3);

	lua_pushstring(L, "QuotaNonPagedPoolUsage");
	lua_pushinteger(L, pmc.QuotaNonPagedPoolUsage);
	lua_settable(L, -3);

	lua_pushstring(L, "QuotaPagedPoolUsage");
	lua_pushinteger(L, pmc.QuotaPagedPoolUsage);
	lua_settable(L, -3);

	lua_pushstring(L, "QuotaPeakNonPagedPoolUsage");
	lua_pushinteger(L, pmc.QuotaPeakNonPagedPoolUsage);
	lua_settable(L, -3);

	lua_pushstring(L, "QuotaPeakPagedPoolUsage");
	lua_pushinteger(L, pmc.QuotaPeakPagedPoolUsage);
	lua_settable(L, -3);

	lua_pushstring(L, "WorkingSetSize");
	lua_pushinteger(L, pmc.WorkingSetSize);
	lua_settable(L, -3);

	//we return one table
	return 1;
}


/*
	This function will retrieve affinity for process specified by its handle.

	Parameters:
		handle - handle of process we want to get affinity for. This has to be greater than 0. We treat that as long.
	Returns:
		processAffinity - affinity mask for the specified process.
		systemAffinity - affinity mask for the system.
	Throws:
		Will throw error if something goes wrong.
*/
int getAffinity(lua_State *L) {
	long argument;
	long handle;
	DWORD processAffinity;
	DWORD systemAffinity;

	//check if given parameter is number
	argument = luaL_checklong(L, 1);
	if (argument < 0) {
		return luaL_error(L, "Invalid argument. A process handle given: %d. Process handle cannot be negative!", argument);
	}
	else {
		//store read argument as process handle
		handle = argument;
	}

	BOOL result = GetProcessAffinityMask((HWND)(LONG_PTR)handle, &processAffinity, &systemAffinity);
	if (result == FALSE) {
		return getLastError(L);
	}

	//now error, store results
	lua_pushinteger(L, processAffinity);
	lua_pushinteger(L, systemAffinity);
	return 2;
}


/*
	This function will set affinity for process specified by its handle.

	Parameters:
		handle - handle of process we want to change affinity for. This has to be greater than 0. We treat that as long.
		affinity - affinity we want to set. This must greater than 0.
	Returns:
		Nothing if setting of affinity succeedes, otherwise there will be an error.
	Throws:
		Will throw error if something goes wrong.
*/
int setAffinity(lua_State *L) {
	long handle;
	long argument;
	DWORD_PTR affinity = 1;

	//check if given parameter is number
	argument = luaL_checklong(L, 1);
	if (argument < 0) {
		return luaL_error(L, "Invalid argument. A process handle given: %d. Process handle cannot be negative!", argument);
	}
	else {
		handle = argument;
	}

	//second argument should be affinity mask
	argument = luaL_checklong(L, 2);
	if (argument < 0) {
		return luaL_error(L, "Invalid argument. A process affinity mask given: %d. Process affinity mask cannot be negative!", argument);
	}
	else {
		affinity = (DWORD_PTR)argument;
	}

	//try to set affinity for process and inform when fail
	BOOL result = SetProcessAffinityMask((HWND)(LONG_PTR)handle, affinity);
	if (result == FALSE) {
		return getLastError(L);
	}

	return 0;
}


/*
	This function will get priority for process specified by its handle.

	Parameters:
		handle - handle of process we want to get priority for. This has to be greater than 0. We treat that as long.
	Returns:
		Priority of the process. Either: P_ABOVE_NORMAL, P_BELOW_NORMAL, P_HIGH, P_IDLE, P_NORMAL, or P_REALTIME
	Throws:
		Will throw error if something goes wrong.
*/
int getPriority(lua_State *L) {
	long handle;
	long argument;

	//check if given parameter is number
	argument = luaL_checklong(L, 1);
	if (argument < 0) {
		return luaL_error(L, "Invalid argument. A process handle given: %d. Process handle cannot be negative!", argument);
	}
	else {
		handle = argument;
	}

	DWORD priority = GetPriorityClass((HWND)(LONG_PTR)handle);
	if (priority == 0) {
		return getLastError(L);
	}

	//return priority
	lua_pushinteger(L, priority);
	return 1;
}


/*
	This function will set affinity for process specified by its handle.

	Parameters:
		handle - handle of process we want to change affinity for. This has to be greater than 0. We treat that as long.
		priority - priority the process should have. One of following constants are supported: P_ABOVE_NORMAL, P_BELOW_NORMAL, P_HIGH, P_IDLE, P_NORMAL, or P_REALTIME
	Returns:
		Nothing if setting of priority succeded, otherwise, there will be an error.
	Throws:
		Will throw error if something goes wrong.
*/
int setPriority(lua_State *L) {
	long handle;
	long argument;
	DWORD priority;

	//check if given parameter is number
	argument = luaL_checklong(L, 1);
	if (argument < 0) {
		return luaL_error(L, "Invalid argument. A process handle given: %d. Process handle cannot be negative!", argument);
	}
	else {
		handle = argument;
	}

	//get priority argument
	priority = luaL_checklong(L, 2);
	switch (priority) {
	case ABOVE_NORMAL_PRIORITY_CLASS:
	case BELOW_NORMAL_PRIORITY_CLASS:
	case HIGH_PRIORITY_CLASS:
	case IDLE_PRIORITY_CLASS:
	case NORMAL_PRIORITY_CLASS:
	case REALTIME_PRIORITY_CLASS:
		break;
	default:
		return luaL_error(L, "The given priority class: %d is NOT supported!", priority);
	};

	//set process priority
	BOOL result = SetPriorityClass((HWND)(LONG_PTR)handle, priority);
	if (result == FALSE) {
		return getLastError(L);
	}

	return 0;
}